#include <opencv2/opencv.hpp>
#include <cmath>

void create_test_images()
{
    int size = 512;
    
    // 1. Checkerboard (high frequency test)
    cv::Mat checkerboard(size, size, CV_8UC3, cv::Scalar(255, 255, 255));
    int square_size = 4;
    for (int y = 0; y < size; y++) {
        for (int x = 0; x < size; x++) {
            if (((x / square_size) + (y / square_size)) % 2 == 0) {
                checkerboard.at<cv::Vec3b>(y, x) = cv::Vec3b(0, 0, 0);
            }
        }
    }
    cv::imwrite("test_checkerboard.png", checkerboard);
    
    // 2. Radial frequency sweep (moiré test)
    cv::Mat radial(size, size, CV_8UC3);
    int cx = size / 2, cy = size / 2;
    for (int y = 0; y < size; y++) {
        for (int x = 0; x < size; x++) {
            double dx = x - cx;
            double dy = y - cy;
            double r = sqrt(dx*dx + dy*dy);
            double freq = r / 2.0;  // frequency increases with radius
            int val = (int)(127.5 * (1.0 + sin(freq)));
            radial.at<cv::Vec3b>(y, x) = cv::Vec3b(val, val, val);
        }
    }
    cv::imwrite("test_radial.png", radial);
    
    // 3. Diagonal lines (aliasing test)
    cv::Mat diagonal(size, size, CV_8UC3, cv::Scalar(255, 255, 255));
    for (int i = 0; i < size; i += 2) {
        cv::line(diagonal, cv::Point(i, 0), cv::Point(0, i), cv::Scalar(0, 0, 0), 1);
        cv::line(diagonal, cv::Point(size-1, i), cv::Point(i, size-1), cv::Scalar(0, 0, 0), 1);
    }
    cv::imwrite("test_diagonal.png", diagonal);
    
    // 4. Smooth gradient (smoothness test)
    cv::Mat gradient(size, size, CV_8UC3);
    for (int y = 0; y < size; y++) {
        for (int x = 0; x < size; x++) {
            int val = (255 * x) / size;
            gradient.at<cv::Vec3b>(y, x) = cv::Vec3b(val, val, val);
        }
    }
    cv::imwrite("test_gradient.png", gradient);
    
    // 5. Sharp edges (sharpness test)
    cv::Mat edges(size, size, CV_8UC3, cv::Scalar(255, 255, 255));
    cv::rectangle(edges, cv::Point(size/4, size/4), 
                  cv::Point(3*size/4, 3*size/4), cv::Scalar(0, 0, 0), -1);
    cv::circle(edges, cv::Point(size/2, size/2), size/6, cv::Scalar(128, 128, 128), -1);
    cv::imwrite("test_edges.png", edges);
    
    // 6. Zone plate (ultimate frequency test)
    cv::Mat zoneplate(size, size, CV_8UC3);
    for (int y = 0; y < size; y++) {
        for (int x = 0; x < size; x++) {
            double dx = (x - cx) / 100.0;
            double dy = (y - cy) / 100.0;
            double r2 = dx*dx + dy*dy;
            int val = (int)(127.5 * (1.0 + sin(r2 * 20.0)));
            zoneplate.at<cv::Vec3b>(y, x) = cv::Vec3b(val, val, val);
        }
    }
    cv::imwrite("test_zoneplate.png", zoneplate);
    
    std::cout << "Created 6 test images (512x512)" << std::endl;
}

int main()
{
    create_test_images();
    return 0;
}
